import { NextRequest, NextResponse } from 'next/server';
import { getSupabaseClient } from '@/storage/database/supabase-client';
import { cookies } from 'next/headers';

// GET /api/admin/orders/by-number/[orderNumber] - 根据订单号获取订单详情（管理员）
export async function GET(
  request: NextRequest,
  { params }: { params: Promise<{ orderNumber: string }> }
) {
  try {
    const { orderNumber } = await params;
    
    // 验证管理员身份
    const cookieStore = await cookies();
    const adminId = cookieStore.get('admin_id')?.value;
    
    if (!adminId) {
      return NextResponse.json(
        { error: '未授权' },
        { status: 401 }
      );
    }
    
    const client = getSupabaseClient();
    
    const { data, error } = await client
      .from('orders')
      .select('*')
      .eq('order_number', orderNumber)
      .single();
    
    if (error || !data) {
      return NextResponse.json(
        { error: '订单不存在' },
        { status: 404 }
      );
    }
    
    return NextResponse.json({ order: data });
  } catch (error) {
    console.error('Error:', error);
    return NextResponse.json(
      { error: '服务器错误' },
      { status: 500 }
    );
  }
}
